/**
* \file: automounter_api_events.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#ifndef AUTOMOUNTER_API_EVENTS_H_
#define AUTOMOUNTER_API_EVENTS_H_

#include "automounter_types.h"

#ifdef __cplusplus
extern "C" {
#endif

typedef struct device_info_t device_info_t;

typedef struct partition_info_t partition_info_t;

/**
 * \brief defines a callback function for connection state change events
 */
typedef void (*connection_state_changed_callback_t)(void);

/**
 * \brief defines a callback function for device state change events
 * \param device_info the info data structure of the device that changed its state
 */
typedef void (*device_state_changed_callback_t)(const device_info_t *device_info);

/**
 * \brief defines a callback function for a device info update event
 * \param device_info the info data structure of the device
 * \param request_id the id of the snapshot request. This id can be used to relate an update event to a snapshot request.
 */
typedef	void (*update_device_info_callback_t)(const device_info_t *device_info, int request_id);

/**
 * \brief defines a callback function for partition state change events
 * \param partition_info the info data structure of the partition
 * \param device_info the info data structure of the parent device of the partition
 */
typedef void (*partition_state_changed_callback_t)(const partition_info_t *partition_info,
		const device_info_t *device_info);

/**
 * \brief defines a callback function for a partition info update event
 * \param partition_info the info data structure of the partition
 * \param device_info the info data structure of the parent device of the partition
 * \param request_id the id of the snapshot request. This id can be used to relate an update event to a snapshot request.
 */
typedef	void (*update_partition_info_callback_t)(const partition_info_t *partition_info,
		const device_info_t *device_info, int request_id);

/**
 * \brief defines a callback function for a snapshot complete event
 * \param request_id the id of the snapshot request that is signaled to be complete with this callback
 */
typedef void (*snapshot_complete_callback_t)(int request_id);

/**
 * \brief Defines events for which applications can register callback functions.
 *
 * Please node: The callbacks are finally called as a result of
 * the application calling automounter_api_dispatch_event(). So finally the thread of the application calling
 * automounter_api_dispatch_event() will be the one calling the event callback.
 */
typedef struct automounter_api_callbacks_t
{
	//--------------- connection events ----------------------------------------------------------------------------

	/**
	 * \brief Called when a connection to the automounter is established.
	 *
	 * This event is fired as a result of a call of automounter_api_try_connect().
	 * The call automounter_api_connect() works synchronously so that the event is not fired
	 * on a call of this function.
	 */
	connection_state_changed_callback_t on_establish_connection_success;

	/**
	 * \brief Called in case of failures while the library is automatically connecting with the automounter.
	 *
	 * The shared library tries to establish a connection to the automounter after a call of
	 * automounter_api_try_connect() once the automounter daemon is up. If problems occurs at this point, the shared
	 * library enters the state AM_API_INITIALIZED_DISCONNECTED and this event is fired.
	 */
	connection_state_changed_callback_t on_establish_connection_failure;

	/**
	 * \brief Called when the shared library looses the connection to the automounter daemon.
	 *
	 * This is normally only the case when the automounter exists. This event is not send if the application closes
	 * the connection itself by calling automounter_api_disconnect().
	 */
	connection_state_changed_callback_t on_connection_lost;

	//--------------- device and partition state changes -----------------------------------------------------------

	/**
	 * \brief Called ones a new mass storage device is detected.
	 */
	device_state_changed_callback_t on_device_detected;

	/**
	 * \brief Called ones the medium is removed from a device or once the automounter detects that a new device does not
	 * contain a medium.
	 */
	device_state_changed_callback_t on_device_nomedia;

	/**
	 * \brief Called when the automounter begins mounting the partitions of a newly detected device.
	 */
	device_state_changed_callback_t on_device_automounting;

	/**
	 * \brief Called after the automounter finished processing all partitions of a newly detected device.
	 */
	device_state_changed_callback_t on_device_automounted;

	/**
	 * \brief Called when the automounter starts unmounting the partitions of a device.
	 */
	device_state_changed_callback_t on_device_unmounting;

	/**
	 * \brief Called after the automounter finished unmounting all mounted partitions of a device.
	 */
	device_state_changed_callback_t on_device_unmounted;

	/**
	 * \brief Called once a device has been removed to indicate that the device is not available any longer.
	 */
	device_state_changed_callback_t on_device_invalid;

	/**
	 * \brief Called once for each partition found on a newly detected device.
	 */
	partition_state_changed_callback_t on_partition_detected;

	/**
	 * \brief Called once the automounter detected that the partition is not supported.
	 * The reason can be found in the unsupported_reason field of the partition data structure.
	 */
	partition_state_changed_callback_t on_partition_unsupported;

	/**
	 * \brief Called once the automounter starts mounting the newly detected partition.
	 */
	partition_state_changed_callback_t on_partition_mounting;

	/**
	 * \brief Called once the automounter finished mounting the newly detected partition.
	 */
	partition_state_changed_callback_t on_partition_mounted;

	/**
	 * \brief Called when the automounter failed mounting the newly detected partition.
	 * The reason can be found in the mount_error_code field of the data structure.
	 */
	partition_state_changed_callback_t on_partition_mount_err;

	/**
	 * \brief Called once the automounter starts remounting a partition.
	 */
	partition_state_changed_callback_t on_partition_remounting;

	/**
	 * \brief Called once the automounter starts unmounting a partition.
	 */
	partition_state_changed_callback_t on_partition_unmounting;

	/**
	 * \brief Called once the automounter finished unmounting a partition.
	 */
	partition_state_changed_callback_t on_partition_unmounted;

	/**
	 * \brief Called several times once a device or medium has been removed to indicate that one or more partitions
	 * are not available any longer.
	 */
	partition_state_changed_callback_t on_partition_invalid;


	//--------------- snapshot events (for getting device and partition information) -------------------------------

	/**
	 * \brief Called to send information about a device to the application.
	 *
	 * This event is send by the shared library as a result of a snapshot request sent by the application by calling
	 * am_api_get_snapshot() if SNAPSHOT_COMPLETE is passed as snapshot scope. This event is sent for any device
	 * currently managed by the automounter.
	 */
	update_device_info_callback_t on_update_device_info;

	/**
	 * \brief Called to send information about a partition to the application.
	 *
	 * This event is sent by the shared library as a result of a snapshot request sent by the application by calling
	 * am_api_get_snapshot(). This event is sent for any partition currently managed by the automounter that fits to
	 * the request scope.
	 */
	update_partition_info_callback_t on_update_partition_info;

	/**
	 * \brief Called to signal that a snapshot request has been processed completely.
	 *
	 * The application can call am_api_get_snapshot() to get information about the devices and partitions currently managed
	 * by the automounter. After calling the function, the shared library calls the on_update_device_info and
	 * on_update_partition_info callbacks several times to provide the information to the application. Finally, the
	 * shared library calls the on_snapshot_complete callback function to signal the application that the request
	 * has been processed completely.
	 */
	snapshot_complete_callback_t on_snapshot_complete;

} automounter_api_callbacks_t;

/**
 * \brief Call to register callback functions for event notification.
 *
 * This function can be used to register a set of callback functions used to receive events from the automounter and
 * the shared library. See the documentation of the data structure automounter_api_callbacks_t for more details on the
 * individual callback functions.
 * Not all callback functions need to be set in the structure. Set callbacks you are not interested in to NULL. The
 * shared library will regard them as unregistered. Passing NULL to this function deregisters all event callbacks
 * ever set before.
 * Please node that the shared library does not fork any internal thread. So finally, any callback called by the library
 * is finally called with the thread that calls the automounter_api_dispatch_event() function.
 * Please node further that the shared library allows only one callback function set for one event. Subsequent calls of
 * this function will override event callbacks registered with any previous call.
 *
 * \param callback_funcs a data structure of callback functions. NULL can be passed to deregister all callbacks
 * registered before.
 */
void automounter_api_register_callbacks(const automounter_api_callbacks_t *callback_funcs);

#ifdef __cplusplus
}
#endif

#endif
